% Copyright (c) 2016 Giampaolo D'Alessandro
% Mathematical Sciences - University of Southampton
% Email: dales@soton.ac.uk
% 
% Permission is hereby granted, free of charge, to any person obtaining a copy
% of this software and associated documentation files (the "Software"), to deal
% in the Software without restriction, including without limitation the rights
% to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
% copies of the Software, and to permit persons to whom the Software is
% furnished to do so, subject to the following conditions:
% 
% The above copyright notice and this permission notice shall be included in
% all copies or substantial portions of the Software.
% 
% THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
% IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
% FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
% AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
% LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
% OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
% SOFTWARE.

function [x,W,P] = DefClencurt(N)
% Function to compute the weights needed to evaluate the integrals
% $\int_{-1}^{x_k} f(x) dx for a function f(x) known on the Gauss-Lobatto
% grid points x_k.
%
% Input
% N (scalar) - The order of the approximation.  The number of nodes is N+1.
%
% Output
% x (vector (N,1)) - The Gauss-Lobatto nodes.
% W (array (N,N)) - The matrix of the weights.  If the functions values are
% stored in a column vector f, then the integrals from -1 to all the other
% nodes are W*f.
% P (array (N,N) - Given f the coefficients of the Chebychev expansion are
% P*f.  This could be useful to interpolate the function f between the
% nodes.  The interpolation is
% alpha = P*F; 
% f_int = @(alpha,x) cos(acos(x)*(0:(length(F)-1)))*alpha;
% Note that the argument x to f_int can be a scalar or a column vector.

% Compute the nodes
x = cos((0:N)'*pi/N);
% Compute the matrix of the weights
M = cos((0:N)'*(0:N)*pi/N);
M(:,1) = 0.5*M(:,1);
M(:,end) = 0.5*M(:,end);
P = 2*M/N;
Q = zeros(N+1);
for k=0:N
  for n=0:N
    Q(k+1,n+1) = integral(@(xi) cos(n*acos(xi)),-1,x(k+1));
  end
end
Q(:,1) = 0.5*Q(:,1);
W = Q*P;
